%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This work by EPFL STI IBI LBNI is licensed under 
% a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
% Based on a work at http://lbni.epfl.ch/.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [varargout] = prepareSurfaceData( varargin )
%PREPARESURFACEDATA   Prepare data inputs for surface fitting.
%
%   [XOUT, YOUT, ZOUT] = PREPARESURFACEDATA(XIN, YIN, ZIN)
%   [XOUT, YOUT, ZOUT, WOUT] = PREPARESURFACEDATA(XIN, YI, ZIN, WIN)
%
%   This function transforms data, if necessary, for the FIT function as
%   follows:
%
%       * For table data, transforms row (YIN) and column (XIN) headers
%       into arrays YOUT and XOUT that are the same size as ZIN. Warn if
%       XIN and YIN are reversed.
%
%       * Return data as columns regardless of the input shapes. Error if
%       the number of elements do not match. Warn if the number of elements
%       match, but the sizes are different.
%
%       * Convert complex to real (remove imaginary parts) and warn.
%
%       * Remove NaN/Inf from data and warn.
%
%       * Convert non-double to double and warn.
%
%   See also: FIT, EXCLUDEDATA.

%   Copyright 2010 The MathWorks, Inc.
%   $Revision: 1.1.6.1 $    $Date: 2010/05/10 16:59:46 $

% Data is converted from complex to real before NaN/Inf are removed. This
% ensures that if a data point is NaN/Inf only in the imaginary component
% then that point is not removed.


% Check the number of inputs is three or four and that the number of
% outputs is the same as the number of inputs.
iAssertNumInputsIsThreeOrFour( nargin );
iAssertNumOutputsEqualsNumInputs( nargin, nargout );

% We are going to take the input data and then push it through a series of
% "filters". We start by taking all the input data.
data = varargin;

% Filter 1: If the data is tabular, then use MESHGRID to expand the row and
% column headers.
[data{:}] = iMeshgrid( data{:} );

% Check that all the inputs have the same number of elements.
iAssertInputsHaveSameNumElements( data{:} );
% ... but warn if the sizes (shapes) of the inputs differ
iWarnIfSizeMismatch( data{:} );

% Filter 2: Convert all the data to columns
[data{:}] = iEnsureColumns( data{:} );
% Filter 3: Remove the imaginary part of any complex values in the data
[data{:}] = iEnsureReal(    data{:} );
% Filter 4: Remove any NaN or Inf elements from the data
[data{:}] = iEnsureFinite(  data{:} );
% Filter 5: Convert any non-double type to double
[data{:}] = iEnsureDouble(  data{:} );

% The output is then just whatever is left of the data.
varargout = data;
end

function iAssertNumInputsIsThreeOrFour( numInputs )
% iAssertNumInputsIsThreeOrFour -- The number of inputs to this function must be
% three or four.
if numInputs < 3
    error( 'curvefit:prepareSurfaceData:notEnoughInputs', ...
        'Not enough input arguments.' );
end
if numInputs > 4
    error( 'curvefit:prepareSurfaceData:tooManyInputs', ...
        'Too many input arguments.' );
end
end

function iAssertNumOutputsEqualsNumInputs( numInputs, numOutputs )
% iAssertNumOutputsEqualsNumInputs -- The number of outputs from this function
% has to match the number of inputs.
if  numInputs ~= numOutputs
    error( 'curvefit:prepareSurfaceData:numOutputsMustEqualNumInputs', ...
        'Number of output arguments must equal number of input arguments.' );
end
end

function [x, y, z, w] = iMeshgrid( x, y, z, w )
% iMeshgrid -- Expand row & column (Y & X) headers if appropriate

% If Y and Y and vectors and Z is a matrix,
if isvector( x ) && isvector( y ) && ismatrix( z ) && ~isvector( z )
    [numZRows, numZColumns] = size( z );
    % ... and if the size of Y matches the number of rows of Z
    % ... and the size of X matches the number of columns of Z
    if numel( y ) == numZRows && numel( x ) == numZColumns
        % ... then "meshgrid" x and y
        [x, y] = meshgrid( x, y );
        
        % ... but if the size of X matches the number of rows of Z
        % ... and the size of Y matches the number of columns of Z
    elseif numel( x ) == numZRows && numel( y ) == numZColumns
        % ... then "meshgrid" x and y in the swapped positions
        [y, x] = meshgrid( y, x );
        % ... and warn
        warning( 'curvefit:prepareSurfaceData:swapXAndYForTable', ...
            'Using X Input for rows and Y Input for columns to match Z Output matrix' );
    end
end
end

function iAssertInputsHaveSameNumElements( x, y, z, w )
% iAssertInputsHaveSameNumElements -- Ensure that all inputs have the same
% number of elements as each other
if numel( x ) ~= numel( y ) || numel( x ) ~= numel( z ) ...
        || (nargin == 4 && numel( x ) ~= numel( w ) )
    error( 'curvefit:prepareSurfaceData:unequalNumel', ...
        'Data sizes are <a href="matlab:helpview( fullfile( docroot,  ''toolbox'', ''curvefit'', ''curvefit.map'' ), ''sftool_compatible_data'');">incompatible</a>.' );
end
end

function iWarnIfSizeMismatch( x, y, z, w )
% iWarnIfSizeMismatch -- Warn the users if the numbers of elements match but the
% size of the arrays are different
if ~isequal( size( x ), size( y ), size( z ) ) ...
        || (nargin == 4 && ~isequal( size( x ), size( w ) ))
    
    warning( 'curvefit:prepareSurfaceData:sizeMismatch', ...
        'Data sizes do not match. Converting all data to column vectors.' );
end
end

function [x, y, z, w] = iEnsureColumns( x, y, z, w )
% iEnsureColumns -- Ensure that the data is arranged in column vectors
x = x(:);
y = y(:);
z = z(:);
if nargin == 4
    w = w(:);
end
end

function [x, y, z, w] = iEnsureFinite( x, y, z, w )
% iEnsureFinite -- Ensure that the data is finite, i.e., remove NaN and Inf from
% the data
isNotFinite = ~isfinite( x ) | ~isfinite( y ) | ~isfinite( z );
if nargin == 4
    isNotFinite = isNotFinite | ~isfinite( w );
end

if any( isNotFinite )
    warning( 'curvefit:prepareSurfaceData:removingNaNAndInf', ...
        'Removing NaN and Inf from data' );
end

x(isNotFinite) = [];
y(isNotFinite) = [];
z(isNotFinite) = [];

if nargin == 4
    w(isNotFinite) = [];
end
end

function [x, y, z, w] = iEnsureReal( x, y, z, w )
% iEnsureReal -- Ensure that the data is real, i.e., check for an imaginary
% component in the inputs and if there is one warn the user and remove it.
if ~isreal( x ) || ~isreal( y ) || ~isreal( z ) || (nargin == 4 && ~isreal( w ))
    warning( 'curvefit:prepareSurfaceData:convertingComplexToReal', ...
        'Using only the real component of complex data.' );
    x = real( x );
    y = real( y );
    z = real( z );
    if nargin == 4
        w = real( w );
    end
end
end

function [x, y, z, w] = iEnsureDouble( x, y, z, w )
% iEnsureDouble -- Ensure that the data is double. If it is not, convert it
% to double and warn the user
if ~iIsDouble( x ) || ~iIsDouble( y ) || ~iIsDouble( z ) || (nargin == 4 && ~iIsDouble( w ))
    warning( 'curvefit:prepareSurfaceData:nonDouble', ...
        'Converting nondouble values to double values.' );
    x = double( x );
    y = double( y );
    z = double( z );
    if nargin == 4
        w = double( w );
    end
end
end

function tf = iIsDouble( vector )
% iIsDouble -- Test a vector to see if it is a double.
tf = isequal( class( vector ), 'double' );
end
